/** <!------------------------------------------------------------------------->
*
*  @file StateManager_generated.c 
*
*  @creationDate 22 November 2024 14:08:31
*
*  @brief Definition of a shared state management with automatic dependency management.
*
*  @author
*
*  @description
*
*  @copyright
*    Copyright 2024, dSPACE GmbH. All rights reserved.
*
*<!-------------------------------------------------------------------------->*/
#include <DsStd.h>
#include "StateManager_generated.h"
#include "BMUtils.h"
#include <stdlib.h>

#ifdef _MSC_VER
/* Disable the MSVC warning that is thrown for assigning dynamic variables as values to an array */
#pragma warning(disable:4204)
#endif

static uint8 initialized = 0;
static uint8 states[5] = {
	 60,  /* CANFunctionBlock Entity (ID: 11339) */
	 1,  /* ApplicationProcess Entity (ID: 11994) */
	 1,  /* BusCfgConfiguredInstance Entity (ID: 10277) */
	 1,  /* BusConfigurationFunctionBlock Entity (ID: 11337) */
	 1,  /* BusCfgConfiguredInstance Entity (ID: 10366) */
};
static uint8 defaultStates[5] = {
	 60,  /* CANFunctionBlock Entity (ID: 11339) */
	 1,  /* ApplicationProcess Entity (ID: 11994) */
	 1,  /* BusCfgConfiguredInstance Entity (ID: 10277) */
	 1,  /* BusConfigurationFunctionBlock Entity (ID: 11337) */
	 1,  /* BusCfgConfiguredInstance Entity (ID: 10366) */
}; /* this array will never change */
static StateTransitionAndCallbackList callbacksAndTransition[5] = {0};

/**************************************************************************************************\
 *** FUNCTION:
 ***     StateManager_RegisterCallback_NoTransitions
 *** 
 *** DESCRIPTION:
 ***     Registers a new callback function for the given state index, without any expected state changes
 ***     If the state of the provided index changes, the callback function is called 
 ***     immediately without checking for an expected state transition
 ***     This function should be only called in the initialization phase, because it dynamically allocates memory.
 *** 
 *** PARAMETERS:
 ***     Type                 Name           Description
 ***     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ***     uint32               index          The index for the state array.
 ***     StateChangedCallback callback       Callback function reference
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void StateManager_RegisterCallback_NoTransitions(uint32 index, StateChangedCallback callback) {
    StateTransitionAndCallbackList* list = &callbacksAndTransition[index];

    /* When there is no State Transition required to be registered extend the list of callbacks along with the number of callbacks */
    if (list->callbackWoStateTransitions == NULL)
    {
        list->callbackWoStateTransitions = malloc(sizeof(CallbacksWoTransitions));
        list->callbackWoStateTransitions->Callback = malloc(sizeof(StateChangedCallback));
        list->callbackWoStateTransitions->NumberOfCallbacks_WO_Transitions = 0;
    }
    else
    {
        list->callbackWoStateTransitions = realloc(list->callbackWoStateTransitions, sizeof(CallbacksWoTransitions) * (list->callbackWoStateTransitions->NumberOfCallbacks_WO_Transitions + 1));
        list->callbackWoStateTransitions->Callback = realloc(list->callbackWoStateTransitions->Callback, sizeof(StateChangedCallback) * (list->callbackWoStateTransitions->NumberOfCallbacks_WO_Transitions + 1));
    }
    list->callbackWoStateTransitions->Callback[list->callbackWoStateTransitions->NumberOfCallbacks_WO_Transitions++] = callback;

    /* call the callback function in case the actual value differs from default value */
    if (states[index] != defaultStates[index])
    {
        callback(index, states[index], defaultStates[index]);
    }
}


/**************************************************************************************************\
 *** FUNCTION:
 ***     StateManager_RegisterCallback_MultipleTransitions
 *** 
 *** DESCRIPTION:
 ***     Registers a new callback function for the given state index, with multiple expected state changes
 ***     If the state of the provided index changes, the callback function is called only if the state change matches one of the expectes state changes 
 ***     immediately without checking for an expected state transition
 ***     This function should be only called in the initialization phase, because it dynamically allocates memory.
 *** 
 *** PARAMETERS:
 ***     Type                 Name                          Description
 ***     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ***     uint32               index                         The index for the state array.
 ***     StateChangedCallback callback                      Callback function reference
 ***     uint8                expectedNumberOfTransitions   the number of state transitions associated with this index
 ***     uint8                expectesStateTransition       Array of all expected state transitions
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void StateManager_RegisterCallback_MultipleTransitions(uint32 index, StateChangedCallback callback, uint8 expectedNumberOfTransitions, uint8 expectesStateTransition[][2]) {
    StateTransitionAndCallbackList* list = &callbacksAndTransition[index];

    /* Register the callbacks that have more than one possible state transitions that are needed to be registered */
    if (list->callbackAndStateTransitions == NULL)
    {
        /* Allocate memory for the first registration of Callback */
        list->Capacity = 1;
        list->callbackAndStateTransitions = malloc(sizeof(CallbackAndStateTransition) * expectedNumberOfTransitions);
        for (uint32 y = 0; y < expectedNumberOfTransitions; y++)
        {
            list->callbackAndStateTransitions[y].Callback = malloc(sizeof(CallbackAndStateTransition) * list->Capacity);
            list->callbackAndStateTransitions[y].StateTransitions = (stateTransitionType*)malloc(sizeof(stateTransitionType));
            list->callbackAndStateTransitions[y].NumberOfCallbacks = 0;
            list->Size++;
        }
        for (uint32 x = 0; x < expectedNumberOfTransitions; x++)
        {
            /* Save the transitions values in a list of structs of old and new transitions */
            list->callbackAndStateTransitions[x].StateTransitions->oldStateTransition = expectesStateTransition[x][0];
            list->callbackAndStateTransitions[x].StateTransitions->newStateTransition = expectesStateTransition[x][1];
            list->callbackAndStateTransitions[x].Callback[list->callbackAndStateTransitions[x].NumberOfCallbacks++] = callback;
        }
    }
    else
    {
        boolean doesTransitionExist;
        for (uint32 i = 0; i < expectedNumberOfTransitions; i++)
        {
            /* See if a transition already exists */
            doesTransitionExist = 0;
            for (uint32 j = 0; j < list->Size; j++)
            {
                if (list->callbackAndStateTransitions[j].StateTransitions->oldStateTransition == expectesStateTransition[i][0]
                    && list->callbackAndStateTransitions[j].StateTransitions->newStateTransition == expectesStateTransition[i][1])
                {
                    /* If transition exists only extend the callback list and save the callback for the particular expected transition*/
                    list->callbackAndStateTransitions[j].Callback = (StateChangedCallback*)realloc(list->callbackAndStateTransitions[j].Callback, sizeof(StateChangedCallback) * (list->callbackAndStateTransitions[j].NumberOfCallbacks + 1));
                    list->callbackAndStateTransitions[j].Callback[list->callbackAndStateTransitions[j].NumberOfCallbacks++] = callback;
                    doesTransitionExist = 1;
                    break;
                }
            }
            if (!doesTransitionExist)
            {
                list->Capacity *= 2;
                /* If transition does not exist extend the callbackAndStateTransitions, save the new expected transition along with the callback*/
                list->callbackAndStateTransitions = (CallbackAndStateTransition *)realloc(list->callbackAndStateTransitions, sizeof(CallbackAndStateTransition) * (list->Size + 1));
                list->callbackAndStateTransitions[list->Size].StateTransitions = (stateTransitionType*)malloc(sizeof(stateTransitionType));
                list->callbackAndStateTransitions[list->Size].Callback = malloc(sizeof(StateChangedCallback) * list->Capacity);
                list->callbackAndStateTransitions[list->Size].NumberOfCallbacks = 0;

                list->callbackAndStateTransitions[list->Size].StateTransitions->oldStateTransition = expectesStateTransition[i][0];
                list->callbackAndStateTransitions[list->Size].StateTransitions->newStateTransition = expectesStateTransition[i][1];
                list->callbackAndStateTransitions[list->Size].Callback[list->callbackAndStateTransitions[list->Size].NumberOfCallbacks++] = callback;
                list->Size++;
            }
        }
    }
    /* call the callback function in case the actual value differs from default value */
    if (states[index] != defaultStates[index])
    {
        callback(index, states[index], defaultStates[index]);
    }
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     StateManager_RegisterCallback_SingleTransition
 *** 
 *** DESCRIPTION:
 ***     Registers a new callback function for the given state index, with one expected state change
 ***     If the state of the provided index changes, the callback function is called 
 ***     after checking for the expected state transition
 ***     This function should be only called in the initialization phase, because it dynamically allocates memory.
 *** 
 *** PARAMETERS:
 ***     Type                 Name           Description
 ***     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ***     uint32               index               The index for the state array.
 ***     StateChangedCallback callback            Callback function reference
 ***     uint8                expectedOldState    the expected previous state necessary for the check
 ***     uint8                expectedNewState    the expected currennt state necessary for the check
 ***
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void StateManager_RegisterCallback_SingleTransition(uint32 index, StateChangedCallback callback, uint8 expectedOldState, uint8 expectedNewState) {
    uint8 Transitions[1][2] ={{expectedOldState, expectedNewState}};
    StateManager_RegisterCallback_MultipleTransitions(index, callback, 1, Transitions);
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     UnregisterCallback
 *** 
 *** DESCRIPTION:
 ***     Unregisters the provided callback function from the given state index.
 *** 
 *** PARAMETERS:
 ***     Type                 Name           Description
 ***     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ***     uint32               index          The index for the state array.
 ***     StateChangedCallback callback       Callback function reference
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void StateManager_UnregisterCallback_SingleTransition(uint32 index, StateChangedCallback callback, uint8 expectedOldState, uint8 expectedNewState) {
    uint8 Transitions[1][2] ={{expectedOldState, expectedNewState}};
    StateManager_UnregisterCallback_MultipleTransitions(index, callback, 1, Transitions);
}

void StateManager_UnregisterCallback_MultipleTransitions(uint32 index, StateChangedCallback callback, uint8 expectedNumberOfTransitions, uint8 expectesStateTransition[][2]) {
    StateTransitionAndCallbackList* list = &callbacksAndTransition[index];
    if (list->callbackAndStateTransitions == NULL) return;

    for (uint32 i = 0; i < expectedNumberOfTransitions; i++)
    {
        /* See if a transition already exists */
        for (uint32 j = 0; j < list->Size; j++)
        {
            if (list->callbackAndStateTransitions[j].StateTransitions->oldStateTransition == expectesStateTransition[i][0]
                && list->callbackAndStateTransitions[j].StateTransitions->newStateTransition == expectesStateTransition[i][1])
            {
                if (list->callbackAndStateTransitions[i].NumberOfCallbacks == 1 &&
                    list->callbackAndStateTransitions[i].Callback[0] == callback)
                {
                    list->callbackAndStateTransitions[i].Callback[0] = 0;
                    list->callbackAndStateTransitions[i].NumberOfCallbacks = 0;
                }
                else
                {
                    for (uint32 k = 0; k < list->callbackAndStateTransitions[i].NumberOfCallbacks; k++)
                    {
                        if (list->callbackAndStateTransitions[i].Callback[k] == callback)
                        {
                            list->callbackAndStateTransitions[i].Callback[k] =
                            list->callbackAndStateTransitions[i].Callback[list->callbackAndStateTransitions[i].NumberOfCallbacks - 1];
                            list->callbackAndStateTransitions[i].NumberOfCallbacks--;
                        }
                    }
                }
            }
        }
    }
}

void StateManager_UnregisterCallback_NoTransitions(uint32 index, StateChangedCallback callback) {
    StateTransitionAndCallbackList* list = &callbacksAndTransition[index];

    if (list->callbackWoStateTransitions == NULL) return;
    if (list->callbackWoStateTransitions->NumberOfCallbacks_WO_Transitions == 1 &&
       list->callbackWoStateTransitions->Callback[0] == callback)
    {
        list->callbackWoStateTransitions->Callback[0] = 0;
        list->callbackWoStateTransitions->NumberOfCallbacks_WO_Transitions = 0;
    }
    else
    {
        for (uint32 i = 0; i < list->callbackWoStateTransitions->NumberOfCallbacks_WO_Transitions; i++)
        {
            if (list->callbackWoStateTransitions->Callback[i] == callback)
            {
                list->callbackWoStateTransitions->Callback[i] = list->callbackWoStateTransitions->Callback[list->callbackWoStateTransitions->NumberOfCallbacks_WO_Transitions - 1];
                list->callbackWoStateTransitions->NumberOfCallbacks_WO_Transitions--;
            }
        }
    }
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     InvokeCallbacks
 *** 
 *** DESCRIPTION:
 ***     Invokes all callbacks for the provided state index.
 *** 
 *** PARAMETERS:
 ***     Type                 Name           Description
 ***     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ***     uint32               index          The index for the state array.
 ***     uint8                newState          The new state
 ***     uint8                oldState          The current state
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
static void InvokeCallbacks(uint32 index, uint8 newState, uint8 oldState) {
    StateTransitionAndCallbackList* cbAndTransitionlist = &callbacksAndTransition[index];

    if (cbAndTransitionlist->Size != 0)
    {
        for (uint32 i = 0; i < cbAndTransitionlist->Size; i++)
        {
            if (cbAndTransitionlist->callbackAndStateTransitions[i].StateTransitions->oldStateTransition == oldState &&
                cbAndTransitionlist->callbackAndStateTransitions[i].StateTransitions->newStateTransition == newState)
            {
                for (uint32 k = 0; k < cbAndTransitionlist->callbackAndStateTransitions[i].NumberOfCallbacks; k++)
                {
                    cbAndTransitionlist->callbackAndStateTransitions[i].Callback[k](index, newState, oldState);
                }
            }
        }
    }
    if (cbAndTransitionlist->callbackWoStateTransitions != NULL && cbAndTransitionlist->callbackWoStateTransitions->NumberOfCallbacks_WO_Transitions != 0)
    {
        for (uint32 i = 0; i < cbAndTransitionlist->callbackWoStateTransitions->NumberOfCallbacks_WO_Transitions; i++)
        {
            cbAndTransitionlist->callbackWoStateTransitions->Callback[i](index, newState, oldState);
        }
    }
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     StateManager_GetState
 *** 
 *** DESCRIPTION:
 ***     Returns the state of the provided state index.
 *** 
 *** PARAMETERS:
 ***     Type                 Name           Description
 ***     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ***     uint32               index          The index for the state array.
 *** 
 *** RETURNS:
 ***     uint8 - The state of the provided index.
\**************************************************************************************************/
uint8 StateManager_GetState(uint32 index) {
    return states[index];
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     StateManager_GetState_SysIntCodeMap
 *** 
 *** DESCRIPTION:
 ***     Returns the state of the provided state index.
 *** 
 *** PARAMETERS:
 ***     Type                 Name           Description
 ***     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ***     uint32               index          The index for the state array.
 ***     void*                state          The argument received when the SysIntCode mapping takes plave
 *** 
 *** RETURNS:
 ***     void - updates the state of the provided index to the pointer address.
\**************************************************************************************************/
void StateManager_GetState_SysIntCodeMap(uint32 index, void* State) {
    *((uint8 *) State) = states[index];
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     StateManager_SetState
 *** 
 *** DESCRIPTION:
 ***     Sets the state of the provided state index to the given value.
 ***     This function should not be called for dependend states, because they are automatically derived from all dependencies.
 *** 
 *** PARAMETERS:
 ***     Type                 Name           Description
 ***     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ***     uint32               index          The index for the state array.
 ***     uint8                state          The new state.
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void StateManager_SetState(uint32 index, uint8 state) {
    uint8 oldState;

    if(state != states[index]) {
        oldState = states[index];
        states[index] = state;
        InvokeCallbacks(index, state, oldState);
    }
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     StateManager_DecrementCounterState
 *** 
 *** DESCRIPTION:
 ***     Decrements the state (counter) of the provided state index by one.
 ***     If state (counter) is zero no more decrement.
 ***     This function should not be called for dependend states, because they are automatically derived from all dependencies.
 *** 
 *** PARAMETERS:
 ***     Type                 Name           Description
 ***     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ***     uint32               index          The index for the state array.
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void StateManager_DecrementCounterState(uint32 index) {
    uint8 oldState = states[index];

    if(oldState > 0) {
        uint8 state = oldState-1;
        states[index] = state;
        InvokeCallbacks(index, state, oldState);
    }
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     StateManager_ResetCounterState
 *** 
 *** DESCRIPTION:
 ***     Reset state (counter) to default value as set during creation.
 ***     This function should not be called for dependend states, because they are automatically derived from all dependencies.
 *** 
 *** PARAMETERS:
 ***     Type                 Name           Description
 ***     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ***     uint32               index          The index for the state array.
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void StateManager_ResetCounterState(uint32 index) {
    uint8 oldState;
    uint8 state = defaultStates[index];

    if(state != states[index]) {
        oldState = states[index];
        states[index] = state;
        InvokeCallbacks(index, state, oldState);
    }
}

/**************************************************************************************************\
 *** FUNCTION:
 ***     StateManager_SetState_Mutex
 *** 
 *** DESCRIPTION:
 ***     Sets the state of the provided state index to the given value during the SysIntCode mapping.
 ***     This function is specific to the CanIf module since the setting of the states in CanIf is Mutex protected.
 ***     This function should not be called for dependend states, because they are automatically derived from all dependencies.
 *** 
 *** PARAMETERS:
 ***     Type                 Name           Description
 ***     ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ***     uint32               index          The index for the state array.
 ***     uint8                state          The new state.
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
UNUSED static uint32 InterruptStatus = 0;
void StateManager_SetState_Mutex(uint32 index, uint8 state) {
    uint8 oldState;

    if(state != states[index]) {
        /* Mutex protected region begins,Obtain lock for Mutex */
        DS_RTOS_INT_SAVE_AND_DISABLE(InterruptStatus);
         oldState = states[index];
         states[index] = state;
         InvokeCallbacks(index, state, oldState);
        /* Mutex protected region ends,Release Mutex lock */
        DS_RTOS_INT_RESTORE(InterruptStatus);
    }
}


/**************************************************************************************************\
 *** FUNCTION:
 ***     StateManager_StateManagerInit
 *** 
 *** DESCRIPTION:
 ***     Initializes the StateManager by registering all automatically generated callbacks.
 *** 
 *** PARAMETERS:
 *** 
 *** RETURNS:
 ***     void
\**************************************************************************************************/
void StateManager_StateManagerInit(void) {
    if(initialized) return;
    initialized = 1;

}

